/*
 * zeus-cad.c - implementation of SPECweb99 GET with Cookie (custom ads)
 * as an ISAPI extension. Optimised for Zeus Web Server 3.3.8 and above
 *
 * (c) Zeus Technology Limited 2000-2002.  All rights reserved.
 * 
 * Copyright in the source code ("the Source Code") listed below in whatever
 * form, whether printed electronic compiled or otherwise, belongs to Zeus
 * Technology Limited ("we"/"us").
 * 
 * If you have entered into a written agreement with us under which the Source
 * Code is licensed to you ("Licence"),  you may not use, sell, license,
 * transfer, copy or reproduce the Source Code in whole or in part or in any
 * manner or form other than in accordance with your Licence.  To do so is
 * strictly prohibited and may be unlawful and a serious criminal offence.
 * 
 * If you have not entered into a Licence, disclosure of the Source Code is
 * made "as is".   You may use the Source Code for non-commercial purposes
 * only.  You may distribute the Source Code to individual third parties for
 * their non-commercial purposes only, but only if (1) you acknowledge our web
 * site as the source of the Source Code and include such acknowledgement and
 * our web address (www.zeus.com) in any copy of the Source Code; (2) the
 * Source Code you distribute is complete and not modified in any way and
 * includes this notice; and (3) you inform such third parties that these
 * conditions apply to them and that they must comply with them.
 * 
 * If you have not entered into a Licence, all express and implied warranties,
 * conditions, terms, undertakings and representations, including without
 * limitation as to quality, performance, fitness for purpose, or
 * non-infringement, are excluded to the fullest extent permitted by law.
 * Neither we nor any other person involved in disclosing the Source Code shall
 * have any liabilities whatsoever to you, howsoever arising, in connection
 * with the Source Code, or its use by you.
 * 
 * If you have not entered into a Licence but would like to do so,  please
 * contact us at pepp@zeus.com.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>
#include <time.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/param.h>
#include <signal.h>
#include <sys/stat.h>
#ifdef OUT_PROCESS
#include <pthread.h>
#endif
#define DEBUG_NAME "CAD: "
#include "debug.h"
#include "zeus-common.h"

typedef struct {
   int cookie_userid;
   int cookie_lastad;
   int demographics;
   char return_cookie[128];
   int return_cookielen;
   int ad_id;
} cad_t;

#define GENDER_MASK    0x30000000
#define AGE_GROUP_MASK 0x0f000000
#define REGION_MASK    0x00f00000
#define INTEREST1_MASK 0x000ffc00
#define INTEREST2_MASK 0x000003ff

typedef struct {
   int demographics;
   int minimum_match;
   time_t expiration_time;
   struct {
      int gender;
      int age_group;
      int region;
      int interest1;
      int interest2;
   } weight;
} cadfile_entry_t;

#define MAX_CADS 360

/* we shadow the User.Personality and Custom.Ads files in memory... */
int user_personality_max = -1;
int *user_personality = NULL;
cadfile_entry_t custom_ads[MAX_CADS];

#ifdef OUT_PROCESS
static pthread_mutex_t cad_lock = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t upf_lock = PTHREAD_MUTEX_INITIALIZER;
#else
/* this is used by the housekeeping module to send a signal after
 * a reset command, in order that the User.Personality and Custom.Ads
 * files will be forced to be re-read */
static char pid_file[MAXPATHLEN];
static int reload_upf = 1;
static int reload_cad = 1;
#endif

#define HTTP_TYPE_LEN "Content-Type: text/html\nContent-Length: "
#define HTTP_OK_TYPE  "HTTP 200 OK\nContent-Type: text/html\n"
#define COOKIE_PART1  "Set-Cookie: found_cookie=Ad_id="
#define COOKIE_PART2  "&Ad_weight="
#define COOKIE_PART3  "&Expired="
#define COOKIE_UPFERR ( COOKIE_PART1 "-1" COOKIE_PART2 "00" COOKIE_PART3 "1" )

#define MAX_CAD_FILESIZE (100 * 1024)

/* Sample User.Personality line:
 * "  100 21404040\n"
 *  ^     ^         ^
 *  0     6        15
 */
#define UPF_LINE_LENGTH 15

/* ----------------------------------------------------------------------
 * Read the User.Personality file into memory. This is called
 * on the first request, and on the next request after a reset
 * command.
 * -------------------------------------------------------------------- */
static int
read_upf( void )
{
   struct stat st;
   int size;
   static int last_size = 0;
   char line[UPF_LINE_LENGTH + 1];
   int upf;
   int i;

   debug( DEBUG_NAME "cad.%d reloading User.Personality\n", getpid() );

   if ( stat( UPF_PATH, &st ) == -1 ) {
      debug( DEBUG_NAME "couldn't stat %s: %s\n", UPF_PATH, strerror( errno ));
      return Failure;
   }
   /* file size / record length = number of entries */
   size = st.st_size / UPF_LINE_LENGTH;
   if ( size != last_size ) {
      user_personality = realloc(user_personality, size*sizeof(int));
      if ( user_personality == NULL ) {
         debug( DEBUG_NAME "couldn't allocate memory for user_personality\n" );
         return Failure;
      }
      last_size = size;
   }
   /* now read the file... */
   if ( (upf=open( UPF_PATH, O_RDONLY )) == -1 ) {
      debug( DEBUG_NAME "couldn't open %s: %s\n", UPF_PATH, strerror( errno ));
      return Failure;
   }
   for (i=0; i<size; i++) {
      if ( zeus_readall( upf, line, UPF_LINE_LENGTH ) < 0 ) {
         close( upf );
         debug( DEBUG_NAME "couldn't read from %s: %s\n",
                UPF_PATH, strerror( errno ));
         return Failure;
      }
      /* got the newline, so no need to terminate */
      user_personality[i] = strtoul(line+6,NULL,16);
   }
   close(upf);
   user_personality_max = size - 1;
   return Success;
}

/* Sample Custom.Ads line:
 * "  100 21404040    7D8C9  75  982621356\n"
 *  ^     ^        ^        ^   ^           ^
 *  0     6       15       24  28          39
 */
#define CAD_LINE_LENGTH 39

/* ----------------------------------------------------------------------
 * Read the Custom.Ads file into memory.  This is called
 * on the first request, and on the next request after a reset
 * command.
 * -------------------------------------------------------------------- */
static int
read_cad(void)
{
   char line[CAD_LINE_LENGTH + 1];
   int cad;
   int i, weight;

   debug( DEBUG_NAME "cad.%d reloading Custom.Ads\n", getpid() );

   /* now read the file... */
   if ( (cad=open( CAD_PATH, O_RDONLY )) == -1 ) {
      debug( DEBUG_NAME "couldn't open %s: %s\n", CAD_PATH, strerror( errno ));
      return Failure;
   }
   for ( i = 0; i < MAX_CADS; i++ ) {
      if ( zeus_readall( cad, line, CAD_LINE_LENGTH ) < 0 ) {
         close(cad);
         debug( DEBUG_NAME "couldn't read from %s: %s\n",
                CAD_PATH, strerror( errno ));
         return Failure;
      }
      custom_ads[i].demographics     = strtoul(line+6,NULL,16);
      weight = strtoul(line+15,NULL,16);
      custom_ads[i].weight.gender    = (weight >> 16) & 0x0000000f;
      custom_ads[i].weight.age_group = (weight >> 12) & 0x0000000f;
      custom_ads[i].weight.region    = (weight >>  8) & 0x0000000f;
      custom_ads[i].weight.interest1 = (weight >>  4) & 0x0000000f;
      custom_ads[i].weight.interest2 =  weight        & 0x0000000f;
      custom_ads[i].minimum_match    = atoi(line+24);
      custom_ads[i].expiration_time  = atoi(line+28);
   }
   close(cad);
   return Success;
}

/* ----------------------------------------------------------------------
 * This is the first step in handling a CAD GET request.
 * The format of a cookie as received by us is:
 *   my_cookie=user_id=[MyUser]&last_ad=[Last_ad]
 * This routine extracts the two keys as numbers.
 * -------------------------------------------------------------------- */
static void
parse_cookie( LPEXTENSION_CONTROL_BLOCK ecb, request_t *request, cad_t *cad )
{
   char cookie[128];
   int len = 128;
#ifdef OUT_PROCESS
   ecb->GetServerVariable( ecb->ConnID, "HTTP_COOKIE", cookie, &len );
#else
   ecb->GetServerVariable( ecb->ConnID, "COOKIE", cookie, &len );
#endif
   cad->cookie_userid = atoi( &cookie[18] );
   cad->cookie_lastad = atoi( &cookie[32] );
}

/* ----------------------------------------------------------------------
 * This is the second step in handling a CAD GET request.
 * Read the User.Personality file, and extract the UserDemographics
 * field for the userid sent in this cookie. Note that the file may
 * change from under us, so check the mtime and re-read as necessary.
 * -------------------------------------------------------------------- */
static int
lookup_demographics( LPEXTENSION_CONTROL_BLOCK ecb,
                     request_t *request, cad_t *cad )
{
   int i;

#ifdef OUT_PROCESS
   static time_t upf_mtime = 0;
   struct stat st;
   if ( stat( UPF_PATH, &st ) == -1 ) {
      debug( DEBUG_NAME "couldn't stat %s: %s\n", UPF_PATH, strerror( errno ));
      return Failure;
   }
   pthread_mutex_lock( &upf_lock );
   if ( st.st_mtime > upf_mtime ) {
      if ( read_upf() == Failure ) {
         /* debugging output done in read_upf() */
         pthread_mutex_unlock( &upf_lock );
         return Failure;
      }
      upf_mtime = st.st_mtime;
   }
   pthread_mutex_unlock( &upf_lock );
#else
   if ( reload_upf ) {
      if ( read_upf() == Failure )
         /* debugging output done in read_upf() */
         return Failure;
      reload_upf = 0;
   }
#endif
   i = cad->cookie_userid - 10000;
   if ( i < 0 || i > user_personality_max ) {
      debug( DEBUG_NAME "cookie userid %d out of range 0 - %d\n",
             i, user_personality_max );
      return Failure;
   }
   cad->demographics = user_personality[i];
   return Success;
}

/* ----------------------------------------------------------------------
 * This is the third step in handling a CAD GET request.
 * Use the demographics and lastad to generate a cookie for reply.
 * We shadow the Custom.Ads file in memory, so re-read as necessary.
 * Fill in the return_cookie field of this request, and also fill in
 * ad_id - this is used when doing text substitution on class 1 and
 * class 2 files.
 * -------------------------------------------------------------------- */
static int
printd( char *dst, int n )
{
   char buf[14];
   int len, i = 0;
   do {
      buf[i++] = n%10;
      n /= 10;
   } while( n );
   len = i;
   do {
      *(dst++) = buf[--i] + '0';
   } while( i );
   return len;
}

static int
generate_return_cookie( LPEXTENSION_CONTROL_BLOCK ecb,
                        request_t *request, cad_t *cad )
{
#ifdef OUT_PROCESS
   static time_t cad_mtime = 0;
   struct stat st;
#else
   static int *time_ptr = NULL;
#endif
   int demographics, weight, ad_index, i;
   int expired = 0, len;

#ifdef OUT_PROCESS
   if ( stat( CAD_PATH, &st ) == -1 ) {
      debug( DEBUG_NAME "couldn't stat %s: %s\n", CAD_PATH, strerror( errno ));
      return Failure;
   }
   pthread_mutex_lock( &cad_lock );
   if ( st.st_mtime > cad_mtime ) {
      if ( read_cad() == Failure ) {
         /* debugging output done in read_cad() */
         pthread_mutex_unlock( &cad_lock );
         return Failure;
      }
      cad_mtime = st.st_mtime;
   }
   pthread_mutex_unlock( &cad_lock );
#else
   if ( time_ptr == NULL ) {
      int sz = sizeof( time_ptr );
      if ( ecb->ServerSupportFunction( ecb->ConnID, HSE_REQ_GET_TIMEPTR,
                                       &time_ptr, &sz, 0 ) == FALSE ) {
         debug( DEBUG_NAME "couldn't get current time pointer from server\n" );
         return Failure;
      }
   }
   if ( reload_cad ) {
      if ( read_cad() == Failure )
         /* debugging output done in read_cad() */
         return Failure;
      reload_cad = 0;
   }
#endif
   /* start searching from lastad + 1... */
   ad_index = cad->cookie_lastad + 1;
   if ( ad_index >= MAX_CADS ) ad_index = 0;
   /* search through the custom.ads file until we hit a match... */
   for ( i = 0; i < MAX_CADS; i++ ) {
      demographics = custom_ads[ad_index].demographics & cad->demographics;
      weight = 0;
      if ( demographics & GENDER_MASK )
         weight += custom_ads[ad_index].weight.gender;
      if ( demographics & AGE_GROUP_MASK )
         weight += custom_ads[ad_index].weight.age_group;
      if ( demographics & REGION_MASK )
         weight += custom_ads[ad_index].weight.region;
      if ( demographics & INTEREST1_MASK )
         weight += custom_ads[ad_index].weight.interest1;
      if ( demographics & INTEREST2_MASK )
         weight += custom_ads[ad_index].weight.interest2;
      if ( weight >= custom_ads[ad_index].minimum_match ) {
#ifdef OUT_PROCESS
         if ( time(NULL) > custom_ads[ad_index].expiration_time )
#else
         if ( *time_ptr > custom_ads[ad_index].expiration_time )
#endif
            expired = 1;
         else
            expired = 0;
         cad->ad_id = ad_index;
         /* build up the cookie string */
         memcpy( cad->return_cookie, COOKIE_PART1, sizeof(COOKIE_PART1) - 1 );
         len = sizeof(COOKIE_PART1) - 1;
         len += printd( cad->return_cookie + len, ad_index );
         memcpy( cad->return_cookie + len, COOKIE_PART2,
                 sizeof(COOKIE_PART2) - 1 );
         len += sizeof(COOKIE_PART2) - 1;
         len += printd( cad->return_cookie + len, weight );
         memcpy( cad->return_cookie + len, COOKIE_PART3,
                 sizeof(COOKIE_PART3) - 1 );
         len += sizeof(COOKIE_PART3) - 1;
         if ( expired )
            cad->return_cookie[len++] = '1';
         else
            cad->return_cookie[len++] = '0';
         cad->return_cookie[len] = '\0';
         cad->return_cookielen = len;
         return Success;
      }
      ad_index++;
      if ( ad_index >= MAX_CADS ) ad_index = 0;
   }

   /* Failed to find a cookie, but the specification requires we carry on
    * anyway.
    */
   debug( DEBUG_NAME "generate_cookie: no match\n" );
   memcpy( cad->return_cookie, COOKIE_PART1, sizeof(COOKIE_PART1) - 1 );
   len = sizeof(COOKIE_PART1) - 1;
   len += printd( cad->return_cookie + len, ad_index );
   memcpy( cad->return_cookie + len, COOKIE_PART2, sizeof(COOKIE_PART2) - 1 );
   len += sizeof(COOKIE_PART2) - 1;
   len += printd( cad->return_cookie + len, weight );
   memcpy( cad->return_cookie + len, COOKIE_PART3, sizeof(COOKIE_PART3) - 1 );
   len += sizeof(COOKIE_PART3) - 1;
   if ( expired )
      cad->return_cookie[len++] = '1';
   else
      cad->return_cookie[len++] = '0';
   cad->return_cookie[len] = '\0';
   cad->return_cookielen = len;
   return Failure;
}

#define CAD_TEMPLATE "<!WEB99CAD><IMG SRC=\"/file_set/dir"
#define CAD_TEMPLATE_LEN (sizeof(CAD_TEMPLATE)-1)

typedef struct {
   char text[16];
} replacement_t;

/* Fast string-searching using the Boyer-Moore algorithm. The array below
 * was generated by looking for the last instance of each ASCII character
 * in CAD_TEMPLATE and taking the index (1-based) of that instance. See
 * chapter 34 of Cormen, Leiserson, and Rivest for more information.
 */
static const char bm[256] = {
    0,  0,  0,  0,  0,  0,  0,  0, /* 000-007 */
    0,  0,  0,  0,  0,  0,  0,  0, /* 008-015 */
    0,  0,  0,  0,  0,  0,  0,  0, /* 016-023 */
    0,  0,  0,  0,  0,  0,  0,  0, /* 024-031 */
   16,  2, 21,  0,  0,  0,  0,  0, /* 032-039 */
    0,  0,  0,  0,  0,  0,  0, 31, /* 040-047 */
    0,  0,  0,  0,  0,  0,  0,  0, /* 048-055 */
    0,  7,  0,  0, 12, 20, 11,  0, /* 056-063 */
    0,  9,  5, 19, 10,  4,  0, 15, /* 064-071 */
    0, 13,  0,  0,  0, 14,  0,  0, /* 072-079 */
    0,  0, 18, 17,  0,  0,  0,  3, /* 080-087 */
    0,  0,  0,  0,  0,  0,  0, 27, /* 088-095 */
    0,  0,  0,  0, 32, 29, 23,  0, /* 096-103 */
    0, 33,  0,  0, 25,  0,  0,  0, /* 104-111 */
    0,  0,  0, 28, 30,  0,  0,  0, /* 112-119 */
    0,  0,  0,  0,  0,  0,  0,  0, /* 120-127 */
    /* Let compiler pad out to 256. */
};

/* ----------------------------------------------------------------------
 * Search for the custom ad template in the supplied buffer and replace
 * it with another string. Returns a pointer to the character after the
 * end of the string, or NULL if there was no match.
 * -------------------------------------------------------------------- */
static char *
search_and_replace( char *buffer, int buflen, const char *replacement )
{
   char *bufptr;
   int i;
   for ( bufptr = buffer + CAD_TEMPLATE_LEN - 1, i = CAD_TEMPLATE_LEN - 1;
	 i >= 0;
	 bufptr--, i-- )
      while ( *bufptr != CAD_TEMPLATE[i] ) {
	 char t = bm[(unsigned char)*bufptr];
	 bufptr += CAD_TEMPLATE_LEN - ((i < t) ? i : t);
	 if ( bufptr - buffer > buflen ) return NULL;
	 i = CAD_TEMPLATE_LEN - 1;
      }
   memcpy( bufptr + CAD_TEMPLATE_LEN + 1, replacement, 14 );
   return bufptr + CAD_TEMPLATE_LEN + 15;
}

/* ----------------------------------------------------------------------
 * This is the final step in handling a CAD GET request.
 * This routine deals with class 1 and class 2 files - it is
 * necessary to perform text substitutions here. Finally the files
 * are transmitted using WriteClient. Note also that
 * 0 <= substitution < MAX_CADS.
 * -------------------------------------------------------------------- */
static void
send_file_with_substitution( LPEXTENSION_CONTROL_BLOCK ecb,
                             request_t *request, cad_t *cad )
{
   in_static char *filebuf = NULL;
   int fd, buflen, len;
   char *bufptr;
   static replacement_t *replacement = NULL;
   in_static char http_header[MEDIUM_BUFFER];

   /* prepare the substitution texts... */
   if ( replacement == NULL ) {
      int i;
      if ( (replacement=malloc(MAX_CADS*sizeof(replacement_t))) == NULL ) {
         isapi_error( ecb, request, DEBUG_NAME "replacement: out of memory" );
         return;
      }
      for ( i = 0; i < MAX_CADS; i++ )
         sprintf( replacement[i].text, "%05d/class%d_%d",
                  i/36, (i%36)/9, i%9 );
   }

   /* create storage space if not already... */
   if ( filebuf == NULL ) {
      if ( (filebuf = malloc( MAX_CAD_FILESIZE )) == NULL ) {
         isapi_error( ecb, request, DEBUG_NAME "filebuf: out of memory" );
         return;
      }
   }
   /* read the file into the buffer... */
   strcpy( path_tail, ecb->lpszQueryString );
   if ( (fd=open( path_root, O_RDONLY )) == -1 ) {
      isapi_error( ecb, request, DEBUG_NAME "couldn't open %s: %s",
                   path_root, strerror( errno ));
      return;
   }
   buflen = zeus_readall( fd, filebuf, MAX_CAD_FILESIZE );
   if ( buflen < 0 ) {
      close( fd );
      isapi_error( ecb, request, DEBUG_NAME "couldn't read from %s: %s",
                   path_root, strerror( errno ));
      return;
   }
   close( fd );
   filebuf[buflen] = '\0';

   /* now make the substitution(s) in the file */
   bufptr = search_and_replace( filebuf, buflen,
		   		replacement[cad->ad_id].text );
   while ( bufptr != NULL )
      bufptr = search_and_replace( bufptr, buflen - (bufptr - filebuf),
				   replacement[cad->ad_id].text );

   /* write out the HTTP headers */
   memcpy( http_header, HTTP_TYPE_LEN, sizeof(HTTP_TYPE_LEN) - 1 );
   len = sizeof(HTTP_TYPE_LEN) - 1;
   len += printd( &http_header[len],
                  request->headerlen + buflen + request->trailerlen );
   http_header[len++] = '\n';
   memcpy( &http_header[len], cad->return_cookie, cad->return_cookielen + 1 );
   len += cad->return_cookielen;
   ecb->ServerSupportFunction( ecb->ConnID, HSE_REQ_SEND_RESPONSE_HEADER,
                               "200 OK", &len, (DWORD*)http_header );

   /* write out the SPEC headers */
   len = request->headerlen;
   ecb->WriteClient( ecb->ConnID, (void*)request->header, &len, 0 );

   /* now write out the file data */
   ecb->WriteClient( ecb->ConnID, (void*)filebuf, &buflen, 0 );

   /* finally write the SPEC trailers */
   len = request->trailerlen;
   ecb->WriteClient( ecb->ConnID, (void*)request->trailer, &len, 0 );

#ifdef OUT_PROCESS
   free( filebuf );
#endif
}

/* ----------------------------------------------------------------------
 * This is the final step in handling a CAD GET request.
 * If the files are neither class 1 nor class 2 then we can simply
 * use TRANSMIT_FILE - this is very efficient. Otherwise we need to
 * do text substitution on the file (see above).
 * -------------------------------------------------------------------- */
static void
send_file( LPEXTENSION_CONTROL_BLOCK ecb, request_t *request, cad_t *cad )
{
   HSE_TF_INFO tf;
   char class;
   in_static char http_header[MEDIUM_BUFFER];

   /* class 1 and class 2 files need text substitution performing... */
   class = ecb->lpszQueryString[24];
   if ( class == '1' || class == '2' ) {
      send_file_with_substitution( ecb, request, cad );
      return;
   }

   /* otherwise use TRANSMIT_FILE to send the file... */
   strcpy( path_tail, ecb->lpszQueryString );
   memcpy( http_header, HTTP_OK_TYPE, sizeof(HTTP_OK_TYPE) - 1 );
   memcpy( http_header + sizeof(HTTP_OK_TYPE) - 1,
           cad->return_cookie, cad->return_cookielen );
   memcpy( http_header + sizeof(HTTP_OK_TYPE) - 1 + cad->return_cookielen,
           "\n", 2 );

   tf.pfnHseIO      = 0;  /* these must be 0 for Zeus */
   tf.pContext      = 0;
   tf.hFile         = (HANDLE) path_root;
   tf.pszStatusCode = http_header;
   tf.BytesToWrite  = 0; /* write all of file */
   tf.Offset        = 0;
   tf.pHead         = (PVOID)request->header;
   tf.HeadLength    = request->headerlen;
   tf.pTail         = (PVOID)request->trailer;
   tf.TailLength    = request->trailerlen;
   /*
    *Because we are not going to be sending any more data after this,
    * we set HSE_IO_DISCONNECT_AFTER_SEND to let the webserver know, and
    * because we are also generating headers, we can use this information
    * to insert a Content-Length & hence use HTTP KeepAlive for better
    * performance.
    */
   tf.dwFlags = HSE_IO_SEND_HEADERS | HSE_IO_DISCONNECT_AFTER_SEND |
      HSE_IO_HANDLE_IS_FILENAME;

#ifdef OUT_PROCESS
   if ( out_process_transmitfile( ecb, request, &tf ) == TRUE ) return;
#else
   if ( ecb->ServerSupportFunction( ecb->ConnID, HSE_REQ_TRANSMIT_FILE,
                                    &tf, 0, 0 ) == TRUE ) return;
#endif

   /* the above failed - abort this spec run! */
   isapi_error( ecb, request,
		DEBUG_NAME "send_file: TRANSMIT_FILE failed: %s", path_root );
}

/* ----------------------------------------------------------------------
 * Processing a GET with custom ads consists of:
 *   o  interpreting the cookie into 'keys'
 *   o  looking up user demographics from the userid key
 *   o  use the demographics info and the lastad key to generate
 *      a new cookie response
 *   o  finally send the new cookie with the requested file. note
 *      that the file may have text substitution applied!
 * -------------------------------------------------------------------- */
void
process_request( LPEXTENSION_CONTROL_BLOCK ecb, request_t *request )
{
   cad_t cad;
   parse_cookie( ecb, request, &cad );
   if ( lookup_demographics( ecb, request, &cad ) == Failure ) {
      debug( DEBUG_NAME "error reading User.Personality\n" );
      cad.ad_id = 0;
      memcpy( cad.return_cookie, COOKIE_UPFERR, sizeof(COOKIE_UPFERR) );
      cad.return_cookielen = sizeof(COOKIE_UPFERR) - 1;
   } else {
      cad.ad_id = 0;
      cad.return_cookielen = 0;
      if ( generate_return_cookie( ecb, request, &cad ) == Failure )
         debug( DEBUG_NAME "error reading Custom.Ads\n" );
   }
   send_file( ecb, request, &cad );
}

/* ----------------------------------------------------------------------
 * The following functions are called when this ISAPI module is
 * loaded / unloaded.
 * -------------------------------------------------------------------- */
#ifndef OUT_PROCESS
void
sigusr2_handler( int how )
{
   reload_upf = 1;
   reload_cad = 1;
}
#endif

int
zeus_isapi_init( void )
{
#ifndef OUT_PROCESS
   int fd, ppid;
   char zeuspid[MAXPATHLEN], pidline[SMALL_BUFFER];
   FILE *zpf;

   sprintf( zeuspid, "%s/web/internal/pid", zeus_home );
   if ( (zpf=fopen(zeuspid,"r")) == NULL ) {
      debug( DEBUG_NAME "Fatal error - couldn't determine webserver pid: %s\n",
	     strerror( errno ));
      return FALSE;
   }
   fgets( pidline, SMALL_BUFFER - 1, zpf );
   pidline[SMALL_BUFFER - 1] = 0;
   fclose( zpf );
   ppid = atoi( pidline );
   sprintf( pid_file, "/tmp/specsignal.%d", ppid );
   fd = open( pid_file, O_WRONLY | O_CREAT | O_APPEND, 0644 );
   if ( fd == -1 ) {
      debug( DEBUG_NAME "Fatal error - couldn't create signal file: %s\n",
             strerror( errno ));
      return FALSE;
   } else {
      char buf[SMALL_BUFFER];
      struct sigaction sa;
      debug( DEBUG_NAME "setting signal handler...\n" );
      sa.sa_handler = sigusr2_handler;
      sigemptyset( &sa.sa_mask );
      sa.sa_flags = 0;
      sigaction( SIGUSR2, &sa, NULL );
      sprintf( buf, "%d\n", getpid() );
      if( zeus_writeall( fd, buf, strlen( buf )) < 0 ) {
         debug( DEBUG_NAME "Fatal error - couldn't write to pid file: %s\n",
                strerror( errno ));
         return FALSE;
      }
      close( fd );
   }
#endif
   return TRUE;
}

void
zeus_isapi_exit( void )
{
#ifndef OUT_PROCESS
   /* Note that all zeus.web child process will race to remove
    * this file - but that's OK as we don't care... at this point
    * we are exiting and we only want to clean up behind ourselves */
   debug( DEBUG_NAME "removing '%s'\n", pid_file );
   unlink( pid_file );
#endif
}
