/*
 * zeus-post.c - implementation of SPECweb99 POST as an ISAPI
 * extension. Optimised for Zeus Web Server 3.3.8 and above
 *
 * (c) Zeus Technology Limited 2000-2002.  All rights reserved.
 * 
 * Copyright in the source code ("the Source Code") listed below in whatever
 * form, whether printed electronic compiled or otherwise, belongs to Zeus
 * Technology Limited ("we"/"us").
 * 
 * If you have entered into a written agreement with us under which the Source
 * Code is licensed to you ("Licence"),  you may not use, sell, license,
 * transfer, copy or reproduce the Source Code in whole or in part or in any
 * manner or form other than in accordance with your Licence.  To do so is
 * strictly prohibited and may be unlawful and a serious criminal offence.
 * 
 * If you have not entered into a Licence, disclosure of the Source Code is
 * made "as is".   You may use the Source Code for non-commercial purposes
 * only.  You may distribute the Source Code to individual third parties for
 * their non-commercial purposes only, but only if (1) you acknowledge our web
 * site as the source of the Source Code and include such acknowledgement and
 * our web address (www.zeus.com) in any copy of the Source Code; (2) the
 * Source Code you distribute is complete and not modified in any way and
 * includes this notice; and (3) you inform such third parties that these
 * conditions apply to them and that they must comply with them.
 * 
 * If you have not entered into a Licence, all express and implied warranties,
 * conditions, terms, undertakings and representations, including without
 * limitation as to quality, performance, fitness for purpose, or
 * non-infringement, are excluded to the fullest extent permitted by law.
 * Neither we nor any other person involved in disclosing the Source Code shall
 * have any liabilities whatsoever to you, howsoever arising, in connection
 * with the Source Code, or its use by you.
 * 
 * If you have not entered into a Licence but would like to do so,  please
 * contact us at pepp@zeus.com.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>
#include <errno.h>
#include <time.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/ipc.h>
#include <sys/sem.h>
#ifdef USE_FLOCK
#include <sys/file.h>
#endif
#define DEBUG_NAME "POST: "
#include "debug.h"
#include "zeus-common.h"

#if ! defined( USE_FLOCK ) && ! defined( USE_LOCKF )
#define USE_SEMOP
#endif

#define MAX_POST_LEN 256

typedef struct {
   char content[MAX_POST_LEN];
   char filename[MAXPATHLEN];
   char mycookie[6];
   char *urlroot;
   char *dir;
   char *class;
   char *num;
   char *client;
} post_t;

#ifdef USE_SEMOP

/* ----------------------------------------------------------------------
 * Updating the POST logfile must be done atomically.
 * These routines are called around the file updating - acquiring
 * and releasing a semaphore. Note that 'union semun' needs to be
 * defined by the user (i.e. here) as mandated by X/OPEN. However, at
 * least one architecture defines union semun, so we need to work
 * around this in Configure.
 * -------------------------------------------------------------------- */
#ifndef HAVE_SEMUN
union semun {
   int val;
   struct semid_ds *buf;
   unsigned short int *array;
};
#endif /* HAVE_SEMUN */

static int semaphore = -1;
static int must_cleanup_semaphore = 0;

#endif /* USE_SEMOP */

static void
acquire_lock( int fd )
{
#if defined( USE_FLOCK )
   int locked = -1;
   while ( locked < 0 )
      locked = flock( fd, LOCK_EX );
#elif defined( USE_LOCKF )
   int retval;
   retval = lockf( fd, F_LOCK, 1 );
   if ( retval != 0 ) {
      debug( DEBUG_NAME "PID %d: couldn't acquire_lock: %s\n",
             getpid(), strerror( errno ) );
      exit( 1 );
   }
#else
   struct sembuf s;
   int i;

   /* create the semaphore if needed (i.e. the first time called) */
   if ( semaphore == -1 ) {
      union semun op;
      key_t key = ftok( "/etc/passwd", 'S' ) + (key_t)getuid();
      semaphore = semget( key, 1, 0600 | IPC_CREAT | IPC_EXCL );
      debug( DEBUG_NAME "PID %d: semget#1 returned %d\n",
             getpid(), semaphore );
      if ( semaphore >= 0 ) {
         /*
          * this was the first instance of requesting a semaphore - the
          * first process successfully created the semaphore. now we
          * need to initialise it...
          */
         op.val = 1;
         if ( semctl( semaphore, 0, SETVAL, op ) == -1 ) {
            debug( DEBUG_NAME
		   "PID %d: semctl failed to initialise semaphore (FATAL): %s\n",
		   getpid(), strerror( errno ));
            exit(1);
         }
         debug( DEBUG_NAME "PID %d: successfully created semaphore\n",
                getpid() );
         must_cleanup_semaphore = 1;
      } else {
         /*
          * we couldn't create the semaphore: perhaps we are another
          * process, and the semaphore already exists?
          */
         struct semid_ds otime_buf;
         int i;
         semaphore = semget( key, 1, 0600 );
         if ( semaphore < 0 ) {
            /* something very bad is going on... */
            debug( DEBUG_NAME "PID %d: semget#2 returned -1 (FATAL)\n",
                   getpid() );
            exit(1);
         }
         debug( DEBUG_NAME "PID %d: found semaphore %d\n",
                getpid(), semaphore );
         /*
          * processes other than that which created the semaphore
          * must wait until the semaphore has been initialised
          * (although in practise this will have already happened
          * during the reset phase of specweb)
          */
         op.buf = &otime_buf;
         /* wait for the first semop() */
         for ( i = 0; i < 10; i++ ) {
            semctl( semaphore, 0, IPC_STAT, op );
            if ( otime_buf.sem_otime != 0 ) {
               debug( DEBUG_NAME "PID %d: otime=0 (%d)\n",
                      getpid(), i );
               break;
            }
            sleep( 1 );
         }
         if ( otime_buf.sem_otime == 0 ) {
            debug( DEBUG_NAME
		   "PID %d: otime stayed 0 while initialising semaphore\n",
		   getpid() );
            exit( 1 );
         }
      }
   }

   /* now try to acquire the lock... */
   s.sem_num = 0;
   s.sem_op = -1;
   s.sem_flg = SEM_UNDO;
   for ( i = 0; i < 1000; i++ ) { /* just in case */
      errno = 0;
      if ( semop( semaphore, &s, 1 ) == -1 ) {
         if ( errno==EAGAIN || errno==ENOSPC || errno==ERANGE ||
              errno==EINTR  || errno==0 )
            continue;
         debug( DEBUG_NAME
                "couldn't perform acquire semaphore operation: %s\n",
		strerror( errno ));
         exit( 1 );
      } else {
         break;
      }
   }
#endif
}

static void
release_lock( int fd )
{
#if defined( USE_FLOCK )
   flock( fd, LOCK_UN );
#elif defined( USE_LOCKF )
   int retval;
   retval = lockf( fd, F_ULOCK, 1 );
   if ( retval != 0 ) {
      debug( DEBUG_NAME "PID %d: couldn't release_lock: %s\n",
             getpid(), strerror( errno ) );
      exit( 1 );
   }
#else
   struct sembuf s;
   int i;
   /* acquire_lock() always either returns a valid semaphore or exits, so
    * no need to check semaphore != -1 here.
    */
   s.sem_num = 0;
   s.sem_op = 1;
   s.sem_flg = SEM_UNDO;
   for ( i = 0; i < 1000; i++ ) { /* just in case */
      errno = 0;
      if ( semop( semaphore, &s, 1 ) == -1 ) {
         if ( errno==EAGAIN || errno==ENOSPC || errno==ERANGE ||
              errno==EINTR  || errno==0 )
            continue;
         debug( DEBUG_NAME
                "couldn't perform release semaphore operation: %s\n",
                strerror( errno ));
         exit( 1 );
      } else {
         break;
      }
   }
#endif
}

/* ----------------------------------------------------------------------
 * This is the first stage in processing a POST request.
 * Read the request from the webserver into a (provided) buffer
 * -------------------------------------------------------------------- */
static int
read_post_request( LPEXTENSION_CONTROL_BLOCK ecb,
                   request_t *request, post_t *post )
{
   char *ptr;
   int to_read;

   to_read = ecb->cbTotalBytes;
   /* allow an extra byte for trailing \0 */
   if ( to_read >= ( MAX_POST_LEN - 1 ) ) {
      isapi_error( ecb, request, DEBUG_NAME "post request too long: %d",
		   to_read );
      return Failure;
   }
   if ( ecb->cbAvailable == to_read ) {
      memcpy( post->content, ecb->lpbData, to_read );
      post->content[to_read] = '\0';
   } else {
      ptr = post->content;
      while( to_read > 0 ) {
         int len;
         len = to_read;
         ecb->ReadClient(ecb->ConnID, ptr, &len);
         to_read -= len;
         ptr += len;
      }
      *ptr = '\0';
   }
   return Success;
}

/* ----------------------------------------------------------------------
 * This is the second stage in processing a POST request.
 * Scan the post data for certain keys, and verify these are all
 * present. Note that the keys can arrive in any order.
 * Finally verify that there was a cookie along with this request,
 * and extract the appropriate part of it into 'mycookie'.
 * -------------------------------------------------------------------- */
static int
parse_post_request( LPEXTENSION_CONTROL_BLOCK ecb,
                    request_t *request, post_t *post )
{
   char cookie[128];
   char *ptr, *next;
   int i, len;

   post->urlroot = NULL;
   post->dir     = NULL;
   post->class   = NULL;
   post->num     = NULL;
   post->client  = NULL;
   ptr = post->content;
   do {
      next = strchr(ptr,'&');
      if ( next ) *(next++) = '\0';

      if      ( strncmp(ptr,"urlroot=",8) == 0 ) post->urlroot = ptr+8;
      else if ( strncmp(ptr,"dir=",    4) == 0 ) post->dir     = ptr+4;
      else if ( strncmp(ptr,"class=",  6) == 0 ) post->class   = ptr+6;
      else if ( strncmp(ptr,"num=",    4) == 0 ) post->num     = ptr+4;
      else if ( strncmp(ptr,"client=", 7) == 0 ) post->client  = ptr+7;

   } while( (ptr=next) != NULL );

   if ( post->urlroot==NULL || post->dir==NULL || post->class==NULL ||
        post->num==NULL || post->client==NULL ) {
      isapi_error( ecb, request, DEBUG_NAME "malformed post request" );
      return Failure;
   }

   /* work out the new cookie to send back in the HTML headers */
   i = 128;
#ifdef OUT_PROCESS
   ecb->GetServerVariable( ecb->ConnID, "HTTP_COOKIE", cookie, &i );
#else
   ecb->GetServerVariable( ecb->ConnID, "COOKIE", cookie, &i );
#endif
   for (i=0; i<5; i++)
      post->mycookie[i] = cookie[i+18];
   post->mycookie[5] = '\0';

   len = strlen( post->urlroot );
   memcpy( post->filename, post->urlroot, len);
   memcpy( &post->filename[len], "dirXXXXX/classX_X", 18 ); /* including \0 */
   memcpy( &post->filename[len+3], post->dir, 5 );
   post->filename[len+14] = post->class[0];
   post->filename[len+16] = post->num[0];
   return Success;
}

/* ----------------------------------------------------------------------
 * This is the third stage in processing a POST request.
 * Write a single line to the post.log file, based on the data that
 * the client just sent to us. Note that the write needs to be atomic.
 * -------------------------------------------------------------------- */
static void
print10d( char *buf, unsigned int n )
{
   char *ptr;
   ptr = buf + 9;
   do {
      *(ptr--) = (n%10) + '0';
      n /= 10;
   } while( n );
   memcpy( buf, "          ", (ptr - buf) + 1 );
}

static int
print10s( char *dst, char *src )
{
   int len = strlen( src );
   if ( len < 10 ) {
      memcpy( dst, "          ", 10-len );
      dst += 10-len;
      memcpy( dst, src, len );
      len = 10;
   } else {
      memcpy( dst, src, len );
   }
   return len;
}

static int
write_post_log( LPEXTENSION_CONTROL_BLOCK ecb,
                request_t *request, post_t *post)
{
#ifndef OUT_PROCESS
   static int *time_ptr = NULL;
#endif
   static pid_t pid = 0;
   static char counter[] = "1234567890 ";
   char buf[MEDIUM_BUFFER];
   int fd, len, i;

#ifndef OUT_PROCESS
   if ( time_ptr == NULL ) {
      int sz = sizeof( time_ptr );
      if ( ecb->ServerSupportFunction( ecb->ConnID, HSE_REQ_GET_TIMEPTR,
                                       &time_ptr, &sz, 0 ) == FALSE ) {
         isapi_error( ecb, request, DEBUG_NAME "couldn't get time_ptr" );
         return Failure;
      }
   }
#endif
   if ( pid == 0 ) pid = getpid();

   if ( (fd=open(POST_LOGFILE,O_RDWR)) == -1 ) {
      isapi_error( ecb, request, DEBUG_NAME "couldn't open post logfile: %s",
		   strerror( errno ));
      return Failure;
   }

   /*
    * we are able to generate all of the response outside a lock, with
    * the exception of the actual counter, which must be read, updated
    * and written atomically - reserve enough space for the counter here
    */
   len = 11;
#ifdef OUT_PROCESS
   print10d( &buf[len], time(NULL) ); len += 10; buf[len++] = ' ';
#else
   print10d( &buf[len], *time_ptr ); len += 10; buf[len++] = ' ';
#endif
   print10d( &buf[len], pid ); len += 10; buf[len++] = ' ';
   for (i=0; i<5; i++)
      buf[len++] = post->dir[i];
   buf[len++] = ' ';
   buf[len++] = ' '; buf[len++] = post->class[0]; buf[len++] = ' ';
   buf[len++] = ' '; buf[len++] = post->num[0];   buf[len++] = ' ';
   len += print10s( &buf[len], post->client ); buf[len++] = ' ';
   for (i=0; i<60; i++) {
      char c;
      c = post->filename[i];
      if ( c == '\0' ) break;
      buf[len++] = c;
   }
   for (; i<60; i++) {
      buf[len++] = ' ';
   }
   buf[len++] = ' ';
   print10d( &buf[len], pid ); len += 10;
   for (i=0; i<6; i++)
      buf[len++] = ' ';
   for (i=0; i<5; i++)
      buf[len++] = post->mycookie[i];
   buf[len++] = '\n'; /* no need for trailing \0 */

   acquire_lock( fd );
   /* read, increment and then write the counter from/to log file */
   zeus_readall( fd, counter, 10 );
   for (i=9; ; ) {
      if ( counter[i] != '9' ) {
         counter[i]++;
         break;
      }
      counter[i--] = '0';
      if ( counter[i] == ' ' ) counter[i] = '0';
   }
   memcpy( buf, counter, 11 );
   lseek( fd, 0, SEEK_SET );
   zeus_writeall( fd, counter, 10 );

   /* append to the log file information on the requested URL */
   lseek(fd, 0, SEEK_END);

   zeus_writeall( fd, buf, len );
   release_lock( fd );
   close( fd );
   return Success;
}

/* ----------------------------------------------------------------------
 * Processing a POST request consists of:
 *   o  receiving the message content from the webserver
 *   o  examining the content and cookie for certain 'keys'
 *   o  writing a line (atomically) to the post logfile based on the keys
 *   o  finally send a response back to the client
 * -------------------------------------------------------------------- */
void
process_request( LPEXTENSION_CONTROL_BLOCK ecb, request_t *request )
{
   HSE_TF_INFO tf;
   in_static char http_header[] = "HTTP 200 OK\nContent-Type: text/html\nSet-Cookie: my_cookie=00000\n";
   post_t post;
   int i;

   /* Debugging output is done in each function, so no need to do it here. */
   if ( read_post_request (ecb, request, &post) == Failure ) return;
   if ( parse_post_request(ecb, request, &post) == Failure ) return;
   if ( write_post_log    (ecb, request, &post) == Failure ) return;

   strcpy( path_tail, post.filename );
   for (i=0; i<5; i++)
      http_header[i+58] = post.mycookie[i];

   tf.pfnHseIO      = 0;  /* these must be 0 for Zeus */
   tf.pContext      = 0;
   tf.hFile         = (HANDLE) path_root;
   tf.pszStatusCode = http_header;
   tf.BytesToWrite  = 0; /* write all of file - could specify a portion of */
   tf.Offset        = 0; /*                     the file to send here      */
   tf.pHead         = (PVOID)request->header;
   tf.HeadLength    = request->headerlen;
   tf.pTail         = (PVOID)request->trailer;
   tf.TailLength    = request->trailerlen;
   /*
    * Because we are not going to be sending any more data after this,
    * we set HSE_IO_DISCONNECT_AFTER_SEND to let the webserver know, and
    * because we are also generating headers, we can use this information
    * to insert a Content-Length & hence use HTTP KeepAlive for better
    * performance.
    */
   tf.dwFlags = HSE_IO_SEND_HEADERS | HSE_IO_DISCONNECT_AFTER_SEND |
      HSE_IO_HANDLE_IS_FILENAME;

#ifdef OUT_PROCESS
   if ( out_process_transmitfile( ecb, request, &tf ) == TRUE ) return;
#else
   if ( ecb->ServerSupportFunction( ecb->ConnID, HSE_REQ_TRANSMIT_FILE,
                                    &tf, 0, 0 ) == TRUE ) return;
#endif

   /* the above failed - abort this spec run! */
   isapi_error( ecb, request,
		DEBUG_NAME "process_request: TRANSMIT_FILE failed: %s",
		path_root );
}

/* ----------------------------------------------------------------------
 * The following functions are called when this ISAPI module is
 * loaded / unloaded. Note that we don't actually need to do anything
 * special in this module.
 * -------------------------------------------------------------------- */
int
zeus_isapi_init( void )
{
   return TRUE;
}

void
zeus_isapi_exit( void )
{
#ifdef USE_SEMOP
   if ( must_cleanup_semaphore ) {
      union semun null_semun = { 0 };
      semctl( semaphore, 0, IPC_RMID, null_semun );
   }
#endif
}
